
#ifndef INCLUDED_GtiHeadMC_h_
#define INCLUDED_GtiHeadMC_h_

#include "Motion/MotionCommand.h"
#include "ERS7.h"

#define TILT 0
#define PAN 1
#define NOD 2

#define TILT_IDX 12
#define PAN_IDX 13
#define NOD_IDX 14

#define SECTOR 	20
#define PAN_MAX 80
#define PAN_MIN -80
#define NOD_MAX 40
#define NOD_MIN -10

#define PAN_ROW1 -15
#define PAN_ROW2 10
#define PAN_ROW3 30

class GtiHeadMC : public MotionCommand {

public:
	GtiHeadMC() : MotionCommand(){
		for (int i = 0; i < 3; i++){
			pendingAngles[i] = 0;
			currentAngles[i] = state->outputs[TILT_IDX + i];
		}
		radPerFrame = 0.01;
		inprogress = false;
		pandir = 1;
		noddir = 1;
	}

	virtual int updateOutputs() {
		
		bool done = true;
		for (int i = 0; i < 3; i++){
			
			bool must_update = false;
			OutputCmd outputCmds[NumFrames];
			
			// Populate the 4 frames for this output
			for (int j = 0; j < (int) NumFrames; j++){
				if (pendingAngles[i] < 0){
					// DBG
					// cout << "updateOutputs(): Joint = " << i <<  " Current = " << currentAngles[i] << " Pending = " << pendingAngles[i] << endl;
					must_update = true;
					if(pendingAngles[i] < - radPerFrame){
						//
						// Check Angle Limit here
						//
						outputCmds[j].set((float)(currentAngles[i] - radPerFrame), 1);
						currentAngles[i] = currentAngles[i] - radPerFrame;
						pendingAngles[i] = pendingAngles[i] + radPerFrame;
					}
					else{
						//
						// Check Angle Limit here
						//
						outputCmds[j].set((float)(currentAngles[i] + pendingAngles[i]), 1);
						currentAngles[i] = currentAngles[i] + pendingAngles[i];
						pendingAngles[i] = 0;
					}
				}
				else if (pendingAngles[i] > 0){
					// DBG
					// cout << "updateOutputs(): Joint = " << i <<  " Current = " << currentAngles[i] << " Pending = " << pendingAngles[i] << endl;
					must_update = true;
					if(pendingAngles[i] > radPerFrame){
						//
						// Check Angle Limit here
						//
						outputCmds[j].set((float)(currentAngles[i] + radPerFrame), 1);
						currentAngles[i] = currentAngles[i] + radPerFrame;
						pendingAngles[i] = pendingAngles[i] - radPerFrame;
					}
					else{
						//
						// Check Angle Limit here
						//
						outputCmds[j].set((float)(currentAngles[i] + pendingAngles[i]), 1);
						currentAngles[i] = currentAngles[i] + pendingAngles[i];
						pendingAngles[i] = 0;
					}
				}
				else{
					outputCmds[j].set((float) currentAngles[i], 1);	
				}
			}
			
			// But update the output ONLY if it has to be moved
			if(must_update){
				// DBG
				// cout << "updateOutputs(): Will update joint " << i << endl;
				done = false;
				inprogress = true;
				motman->setOutput(this, TILT_IDX + i, outputCmds);
			}
		
		}
		
		if(done && inprogress){
			postEvent(EventBase(EventBase::motmanEGID, getID(), EventBase::statusETID));
			// cout << "updateOutputs(): Posted event" << endl;
			inprogress = false;
		}
				
		return 1;
	}
	
	virtual int isDirty() {
		return true;
	}
	
	virtual int isAlive() {
		return true;
	}
	
	void scanSector(){
		
		double x = state->outputs[PAN_IDX] + pandir * (SECTOR * 3.1416 / 180);
		if((x > (PAN_MAX * 3.1416 / 180)) || (x < (PAN_MIN * 3.1416 / 180))){
			pandir = - pandir;
			
			double y = state->outputs[NOD_IDX];
			if( fabs(15 - deg(y)) < 8){
				// Close the to 15 mark: Should go to the 0 mark
				pendingAngles[NOD] = rad(0) - y;
				noddir = -1;
			}
			if( fabs(0 - deg(y)) < 8){
				// Close the to 0 mark: Should go to either 15 or -15 
				// marks depends on noddir
				if(noddir > 0){
					pendingAngles[NOD] = rad(15) - y;
				}
				else{
					pendingAngles[NOD] = rad(-15) - y;
				}	
			}
			else if( fabs(-15 - deg(y)) < 8){
				// Close the to -15 mark: Should go to the 0 mark
				pendingAngles[NOD] = rad(0) - y;
				noddir = 1;
			}
			
		}
		pendingAngles[PAN] = pandir * rad(SECTOR);
		
		// DBG
		// cout << "scanSector(): Current " << state->outputs[PAN_IDX] << ", " << state->outputs[NOD_IDX] << " Pending " << pendingAngles[PAN] << ", " << pendingAngles[NOD] << endl;
		
	}
	
	void pan(int ang){
		pendingAngles[PAN] = rad(ang);	
	}
	
	void nod(int ang){
		pendingAngles[NOD] = rad(ang);	
	}
	
	void tilt(int ang){
		pendingAngles[TILT] = rad(ang);	
	}
	
	void getReady(){
		currentAngles[TILT] = state->outputs[TILT_IDX];
		currentAngles[PAN] = state->outputs[PAN_IDX];
		currentAngles[NOD] = state->outputs[NOD_IDX];
		pendingAngles[TILT] = 0 - currentAngles[TILT];
		pendingAngles[PAN] = (PAN_MIN * 3.1416 / 180) - currentAngles[PAN];
		pendingAngles[NOD] = (NOD_MIN * 3.1416 / 180) - currentAngles[NOD];
		pandir = 1;
		noddir = 1;
		
		// DBG
		for(int i = 0; i < 3; i++){
			cout << "GetReady: Joint = " << i << " Curr = " << currentAngles[i] << " Pend = " << pendingAngles[i] << endl;
		}
		
	}
	
	void updateCurrentAngles(){
		for (int i = 0; i < 3; i++){
			currentAngles[i] = state->outputs[TILT_IDX + i];
		}	
	}
	
	double getCurrentAngle(int joint){
		return currentAngles[joint];	
	}
	
	double deg(double r){
		return r * 180 / 3.1416;	
	}
	
	double rad(double d){
		return d * 3.1416 / 180;	
	}
	
private:
	double pendingAngles[3]; // Rad
	double currentAngles[3]; // Rad
	double radPerFrame;
	int pandir;
	int noddir;
	bool inprogress;
};

#endif
