/********************************************************************
             A Golog/Prolog - Tekkotsu Interface
                        August, 2006.                 

            Do not distribute without permission.
            Include this notice in any copy made.

Permission to use, copy, and modify this software and its documentation 
for non-commercial research purpose is hereby granted without fee,
provided that this permission notice appears in all copies. This
software cannot be used for commercial purposes without written permission.
This software is provided "as is" without express or implied warranty
(including all implied warranties of merchantability and fitness).
No liability is implied for any damages resulting from 
or in connection with the use or performance of this software.

E-mail questions/comments about the interface to Huy Pham:
            hpham [at] scs [dot] ryerson [dot] ca   

Implementation environment:
This interface requires Eclipse Prolog:
	http://eclipse.crosscoreop.com/eclipse/index.html

**********************************************************************/



//////////////////////////////////////////////////////////////////////
//
// aibopred.c
//
// External predicates for Aibo control
//
// To be used with Eclipse Prolog only
// 
//
//
//////////////////////////////////////////////////////////////////////




#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <time.h>
#include "eclipse.h"
#include "TCPComm.h"
#include "ERS7.h"


// Time moment at which we our world begins
time_t zero_time;

// Are we connected to the server? 
int connected = 0;

// Socket used to connect to control server 
int sockfd = 0;


//
// Reset zero_time to current time
//
int reset_time(){
	
	zero_time = time(NULL);
	return PSUCCEED;
}

//
// establish_connection()
// 
// Establish a TCP connection to AIBO control server
// Return 0 on success, -1 on error
//
int establish_connection(){
	
	struct sockaddr_in their_addr;
	struct hostent *he;
	
	//
	// Machine which runs the control server process need to be 
	// DNS lookup-able under the name "aibo" 
	//
	if ((he=gethostbyname("aibo")) == NULL) {
		perror("gethostbyname");
		return(-1);
	}
	
	their_addr.sin_family = AF_INET;    
	their_addr.sin_port = htons(GTI_PORT);  
	their_addr.sin_addr = *((struct in_addr *)he->h_addr);
	memset(&(their_addr.sin_zero), '\0', 8);  

	// socket()/connect()
	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1){
		perror("socket() failed.");
		return(-1);
	}
	if (connect(sockfd, (struct sockaddr *)&their_addr, sizeof(struct sockaddr)) == -1) {
		perror("connect() failed.");
		return(-1);
	}

	// We are now connected!
	connected = 1;
	return 0;
	
}


int query_sensors(){

	char *s;
	pword list, head, tail, arg;
	double num;
	int i;

	struct CmdHdr cmdhdr;
	int sensors[ERS7_NUM_OUTPUTS + ERS7_NUM_SENSORS + ERS7_NUM_BUTTONS];
	double sensorVals[ERS7_NUM_OUTPUTS + ERS7_NUM_SENSORS + ERS7_NUM_BUTTONS];
	int sensorcnt = 0;
	int time;


	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}
	
	// Parse the sensor list
	list = ec_arg(1);
	while (ec_get_list(list, &head, &tail) == PSUCCEED){

		if (ec_get_string(head, &s) != PSUCCEED)
			return PFAIL;
		
		// Look up the name from the three lists: Outputs, Sensor, Buttons
		sensors[sensorcnt] = -1;
		for(i = 0; i < ERS7_NUM_OUTPUTS; i++){
			if (strcmp(s, ERS7_OUTPUTS[i]) == 0){
				sensors[sensorcnt] = i;
				break;
			}
		}
		if (sensors[sensorcnt] < 0){
			for(i = 0; i < ERS7_NUM_SENSORS; i++){
				if (strcmp(s, ERS7_SENSORS[i]) == 0){
					sensors[sensorcnt] = i + ERS7_NUM_OUTPUTS;
					break;
				}
			}
		}
		if (sensors[sensorcnt] < 0){
			for(i = 0; i < ERS7_NUM_BUTTONS; i++){
				if (strcmp(s, ERS7_BUTTONS[i]) == 0){
					sensors[sensorcnt] = i + ERS7_NUM_OUTPUTS + ERS7_NUM_SENSORS;
					break;
				}
			}
		}
		if (sensors[sensorcnt] < 0){
			printf("Cannot lookup sensor index\n");
			return PFAIL;
		}
		
		sensorcnt++;
		list = tail;
	}
	
	// Parse the time
	arg = ec_arg(3);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;
	
	// Cmd header to be sent
	cmdhdr.type = CMDHDR_READ;
	cmdhdr.len = sensorcnt * sizeof(struct SensorCmd);
	
	// Send the header now
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;
		
	// Wait for the right moment and send the body
	if (send(sockfd, sensors, sensorcnt * sizeof(struct SensorCmd), 0) < 0)
		return PFAIL;
		
	// Get the sensor values 
	if (recv(sockfd, sensorVals, sensorcnt * sizeof(double), 0) < 0)
		return PFAIL;

	// DBG
	printf("Query Sensors: returning\n");
	// DBG
	printf("Query Sensors: returning\n");
	
	return ec_unify(ec_arg(2), ec_listofdouble(sensorcnt, sensorVals));
	
}


int query_ball(){

	pword arg;
	double num;
	
	struct CmdHdr cmdhdr;
	int time;
	int ball_index;
	struct Ball ball;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	arg = ec_arg(1);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	ball_index = num;
	
	arg = ec_arg(6);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_QUERYBALL;
	cmdhdr.len = sizeof(int);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait for the right moment and send the body
	if (send(sockfd, &ball_index, sizeof(int), 0) < 0)
		return PFAIL;
	
	// Get the Ball structure
	if (recv(sockfd, &ball, sizeof(struct Ball), 0) < 0)
		return PFAIL;

	ec_unify(ec_arg(2), ec_long((long) ball.visible));
	ec_unify(ec_arg(3), ec_double(ball.xcoord));
	ec_unify(ec_arg(4), ec_double(ball.ycoord));
	ec_unify(ec_arg(5), ec_double(ball.area));
	
	return PSUCCEED;
	
}


int search_ball(){
	pword arg;
	double num;
	
	struct CmdHdr cmdhdr;
	int time;
	int ball_index, result;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	arg = ec_arg(1);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	ball_index = num;
	
	arg = ec_arg(3);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;
	
	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_SEARCHBALL;
	cmdhdr.len = sizeof(int);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait for the right moment and send the body
	if (send(sockfd, &ball_index, sizeof(int), 0) < 0)
		return PFAIL;
	
	// Get the search result
	if (recv(sockfd, &result, sizeof(int), 0) < 0)
		return PFAIL;

	// DBG
	printf("SearchBall result: %d\n", result);
	
	ec_unify(ec_arg(2), ec_long((long) result));
	
	// DBG
	printf("search_ball: returning\n");
	
	return PSUCCEED;
}


int post_joint_cmds(){

	int i;
	char *s;
	pword arg;
	double num;
	pword list, head, tail;
	pword lst, hd, tl;
	int sig;

	struct CmdHdr cmdhdr;
	struct JointCmd cmdbody[ERS7_NUM_OUTPUTS]; 
	int cmdcnt = 0;
	int time;

	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}
	
	// Parse the cmd list, which should look like this:
	//      move([[j1, a1], ...,[jn, an]], t)
	list = ec_arg(1);
	while (ec_get_list(list, &head, &tail) == PSUCCEED){
		
		// Joint name
		if (ec_get_list(head, &hd, &tl) != PSUCCEED)
			return PFAIL;
		if (ec_get_string(hd, &s) != PSUCCEED)
			return PFAIL;

		// Joint index
		cmdbody[cmdcnt].joint = -1;
		for(i = 0; i < ERS7_NUM_OUTPUTS; i++){
			if (strcmp(s, ERS7_OUTPUTS[i]) == 0){
				cmdbody[cmdcnt].joint = i;
				break;
			}
		}
		if (cmdbody[cmdcnt].joint < 0){
			printf("Cannot lookup joint index\n");
			return PFAIL;
		}	
		
		// Joint angle
		lst = tl;
		if (ec_get_list(lst, &hd, &tl) != PSUCCEED)
			return PFAIL;
		if (ec_get_double(hd, &num) != PSUCCEED)
			return PFAIL;
		cmdbody[cmdcnt].angle = num;
		
		// DBG
		printf("post_joint_cmds: Name = %s, Index = %d, Angle = %d\n", s, cmdbody[cmdcnt].joint, cmdbody[cmdcnt].angle);
				
		cmdcnt++;
		list = tail;
		
	}
							
	// Parse the time
	arg = ec_arg(2);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_MOVE;
	cmdhdr.len = cmdcnt * sizeof(struct JointCmd);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, cmdbody, cmdhdr.len, 0) < 0)
		return PFAIL;
	
	// Wait for the recieving ack signal
	if (recv(sockfd, &sig, sizeof(int), 0) < 0)
		return PFAIL;
	
	// DBG
	printf("post_joint_cmds: returning\n");
	
	return PSUCCEED;
	
}


int post_motion_cmd(){

	int i;
	pword arg;
	double num;
	char *s;
	
	struct CmdHdr cmdhdr;
	char motfilename[12];
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	arg = ec_arg(1);
	if (ec_get_string(arg, &s) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(2);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_MSEQ;
	cmdhdr.len = sizeof(char) * strlen(s) + 1; 
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, s, cmdhdr.len, 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}

int post_sound_cmd(){

	int i;
	pword arg;
	double num;
	char *s;
	
	struct CmdHdr cmdhdr;
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	arg = ec_arg(1);
	if (ec_get_string(arg, &s) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(2);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_SOUND;
	cmdhdr.len = sizeof(char) * strlen(s) + 1; 
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, s, cmdhdr.len, 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}

int post_walk_cmd(){

	int i;
	pword arg;
	int sig;
	
	double num;
	struct CmdHdr cmdhdr;
	struct WalkParam wp; 
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	wp.type = WALKTYPE_FIXED;	// we will walk to a fixed location
	
	arg = ec_arg(1);
	if (ec_get_double(arg, &wp.dx) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(2);
	if (ec_get_double(arg, &wp.dy) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(3);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_WALK;
	cmdhdr.len = sizeof(struct WalkParam);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, &wp, sizeof(struct WalkParam), 0) < 0)
		return PFAIL;
	
	// Wait till the completion signal
	if (recv(sockfd, &sig, sizeof(int), 0) < 0)
		return PFAIL;
	
	// DBG
	printf("post_walk_cmd: returning\n");
	
	return PSUCCEED;
	
}


int post_startwalk_cmd(){

	int i;
	pword arg;
	
	double num;
	struct CmdHdr cmdhdr;
	struct WalkParam wp; 
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	wp.type = WALKTYPE_START;  // Start walking indefinitly
	
	arg = ec_arg(1);
	if (ec_get_double(arg, &wp.dx) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(2);
	if (ec_get_double(arg, &wp.dy) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(3);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_WALK;
	cmdhdr.len = sizeof(struct WalkParam);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, &wp, sizeof(struct WalkParam), 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}


int post_endwalk_cmd(){

	int i;
	pword arg;
	
	double num;
	struct CmdHdr cmdhdr;
	struct WalkParam wp; 
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	wp.type = WALKTYPE_END;	
	
	arg = ec_arg(1);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_WALK;
	cmdhdr.len = sizeof(struct WalkParam);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, &wp, sizeof(struct WalkParam), 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}


int post_turn_cmd(){

	int i;
	pword arg;
	int sig;
	
	double num;
	struct CmdHdr cmdhdr;
	struct TurnParam tp; 
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	tp.type = TURNTYPE_FIXED;	// we will turn a fixed angle
	
	arg = ec_arg(1);
	if (ec_get_double(arg, &tp.da) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(2);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_TURN;
	cmdhdr.len = sizeof(struct TurnParam);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, &tp, sizeof(struct TurnParam), 0) < 0)
		return PFAIL;
	
	// DBG
	printf("post_turn_cmd: returning\n");
	
	// Wait till the completion signal
	if (recv(sockfd, &sig, sizeof(int), 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}


int post_startturn_cmd(){

	int i;
	pword arg;
	
	double num;
	struct CmdHdr cmdhdr;
	struct TurnParam tp; 
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	tp.type = TURNTYPE_START;  // Start walking indefinitly
	
	arg = ec_arg(1);
	if (ec_get_double(arg, &tp.da) != PSUCCEED)
		return PFAIL;
	
	arg = ec_arg(2);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_TURN;
	cmdhdr.len = sizeof(struct TurnParam);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, &tp, sizeof(struct TurnParam), 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}


int post_endturn_cmd(){

	int i;
	pword arg;
	
	double num;
	struct CmdHdr cmdhdr;
	struct TurnParam tp; 
	int time;
	
	// Connect to control server
	if (!connected){
		if (establish_connection() < 0){
			return PFAIL;
		}
	}

	tp.type = TURNTYPE_END;	
	
	arg = ec_arg(1);
	if (ec_get_double(arg, &num) != PSUCCEED)
		return PFAIL;
	time = num;

	// Prepare cmd hdr	
	cmdhdr.type = CMDHDR_TURN;
	cmdhdr.len = sizeof(struct TurnParam);
	
	// Send the header now 
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	// Wait until the right time and send the body
	if (send(sockfd, &tp, sizeof(struct TurnParam), 0) < 0)
		return PFAIL;
	
	return PSUCCEED;
	
}


int gti_debug(){
	
	int i;
	pword arg;
	
	double num;
	struct CmdHdr cmdhdr;	
	
	cmdhdr.type = CMDHDR_DEBUG;
	cmdhdr.len = 0;
	
	if (send(sockfd, &cmdhdr, sizeof(struct CmdHdr), 0) < 0)
		return PFAIL;

	return PSUCCEED;
	
}

